param(
    [Parameter(Mandatory=$false, Position=0)]
    [string]$ServerInstance = "localhost",
    [Parameter(Mandatory=$true, Position=1)]
    [string]$Database,
    [Parameter(Mandatory=$false, Position=2)]
    [string]$BackupFolder = "data",
    [Parameter(Mandatory=$false, Position=3)]
    [string]$Username = "",
    [Parameter(Mandatory=$false, Position=4)]
    [string]$Password = "",
    [Parameter(Mandatory=$false)]
    [array]$Views = @(),
    [Parameter(Mandatory=$false)]
    [array]$Functions = @(),
    [Parameter(Mandatory=$false)]
    [array]$Tables = @(),
    [Parameter(Mandatory=$false)]
    [array]$Procedures = @(),
    [Parameter(Mandatory=$false)]
    [array]$Types = @(),
    [Parameter(Mandatory=$false)]
    [ValidateSet("Export", "Import")]
    [string]$Action = "Export"
)

function Import-SqlObjects {
    param (
        [string]$FilePath
    )

    Write-Host "`nImporting SQL objects from $FilePath..."

    $sqlContent = Get-Content -Path $FilePath -Raw
    
    # Split by GO statements (case-insensitive, with optional whitespace)
    $sqlCommands = $sqlContent -split '(?i)\r?\n\s*GO\s*\r?\n'

    Write-Host "Found $($sqlCommands.Count) SQL batches to execute."
    
    $index = 0
    foreach ($sqlCommand in $sqlCommands) {
        $trimmedCommand = $sqlCommand.Trim()
        
        # Skip empty commands
        if (-not [string]::IsNullOrWhiteSpace($trimmedCommand)) {
            try {
                $cmd = $connection.CreateCommand()
                $cmd.CommandText = $trimmedCommand
                $cmd.ExecuteNonQuery()
                Write-Host "Executed SQL batch $($index) successfully"
            }
            catch {
                Write-Error "Failed to execute SQL batch $($index): $($_.Exception.Message)"
                # Write-Host "Command: $trimmedCommand"
            }
        }

        $index++
    }

    Write-Host "Import completed for $FilePath"
}

# --- Helper function: run query and export results ---
function Export-SqlObjects {
    param (
        [string]$Query,
        [string]$ObjectType,
        [string]$FileExtension = "sql"
    )

    Write-Host "`n$($ObjectType):"

    $cmd = $connection.CreateCommand()
    $cmd.CommandText = $Query
    $reader = $cmd.ExecuteReader()

    $allObjects = @()
    while ($reader.Read()) {
        $schema = $reader["SchemaName"]
        $name = $reader["ObjectName"]
        $definition = $reader["Definition"]

        $header = "-- ==========================================
-- Type: $ObjectType
-- Name: [$schema].[$name]
-- Exported on: $(Get-Date)
-- ==========================================

"
        $allObjects += ($header + $definition) + "`nGO`n"
        Write-Host "  $schema.$name"
    }
    $reader.Close()

    if ($allObjects.Count -gt 0) {
        $fileName = "$ObjectType.$FileExtension"
        if (-not (Test-Path $BackupFolder)) {
            New-Item -ItemType Directory -Path $BackupFolder | Out-Null
        }
        $filePath = Join-Path $BackupFolder $fileName
        $content = $allObjects -join "`n"
        Set-Content -Path $filePath -Value $content -Encoding UTF8
        Write-Host "Saved to: $filePath"
    }
}

Write-Host "`nStarting SQL $Action operation on database '$Database' at server '$ServerInstance'..."

# --- Connect to SQL Server ---
if ($Username -and $Password) {
    $connectionString = "Server=$ServerInstance;Database=$Database;User ID=$Username;Password=$Password;TrustServerCertificate=True;"
} else {
    $connectionString = "Server=$ServerInstance;Database=$Database;Integrated Security=True;TrustServerCertificate=True;"
}

Add-Type -AssemblyName "System.Data"
$connection = New-Object System.Data.SqlClient.SqlConnection $connectionString
$connection.Open()

if ($Action -eq "Import") {
    $sqlFiles = Get-ChildItem -Path $BackupFolder -Filter "*.sql"
    foreach ($file in $sqlFiles) {
        Import-SqlObjects -FilePath $file.FullName
    }
} elseif ($Action -eq "Export") {
# --- Stored Procedures ---
if ($Procedures -and $Procedures.Count -gt 0) {
    Export-SqlObjects -Query @"
SELECT 
    SCHEMA_NAME(p.schema_id) AS SchemaName,
    p.name AS ObjectName,
    m.definition AS Definition
FROM sys.procedures p
JOIN sys.sql_modules m ON p.object_id = m.object_id
WHERE p.name IN ('$(($Procedures -join "','"))')
ORDER BY SchemaName, ObjectName
"@ -ObjectType "Procedures"
}

# --- Functions ---
if ($Functions -and $Functions.Count -gt 0) {
Export-SqlObjects -Query @"
SELECT 
    SCHEMA_NAME(f.schema_id) AS SchemaName,
    f.name AS ObjectName,
    m.definition AS Definition
FROM sys.objects f
JOIN sys.sql_modules m ON f.object_id = m.object_id
WHERE f.type IN ('FN','TF','IF') AND f.name IN ('$(($Functions -join "','"))')
ORDER BY SchemaName, ObjectName
"@ -ObjectType "Functions"
}

# --- Views ---
if ($Views -and $Views.Count -gt 0) {
Export-SqlObjects -Query @"
SELECT 
    SCHEMA_NAME(v.schema_id) AS SchemaName,
    v.name AS ObjectName,
    m.definition AS Definition
FROM sys.views v
JOIN sys.sql_modules m ON v.object_id = m.object_id
WHERE v.name IN ('$(($Views -join "','"))')
ORDER BY SchemaName, ObjectName
"@ -ObjectType "Views"
}

# --- Tables ---
if ($Tables -and $Tables.Count -gt 0) {
Export-SqlObjects -Query @"
SELECT 
    SCHEMA_NAME(t.schema_id) AS SchemaName,
    t.name AS ObjectName,
    'CREATE TABLE [' + SCHEMA_NAME(t.schema_id) + '].[' + t.name + '] (' + CHAR(13) + CHAR(10) +
    (
        SELECT 
            '    [' + c.name + '] ' + 
            UPPER(tp.name) +
            CASE 
                WHEN tp.name IN ('varchar', 'char', 'varbinary', 'binary', 'nvarchar', 'nchar')
                    THEN '(' + 
                        CASE WHEN c.max_length = -1 THEN 'MAX' ELSE CAST(
                            CASE WHEN tp.name LIKE 'n%' THEN c.max_length / 2 ELSE c.max_length END AS VARCHAR(5)
                        ) END + ')'
                WHEN tp.name IN ('decimal', 'numeric')
                    THEN '(' + CAST(c.precision AS VARCHAR(5)) + ',' + CAST(c.scale AS VARCHAR(5)) + ')'
                ELSE ''
            END +
            CASE WHEN c.is_nullable = 0 THEN ' NOT NULL' ELSE ' NULL' END + ',' + CHAR(13) + CHAR(10)
        FROM sys.columns c
        JOIN sys.types tp ON c.user_type_id = tp.user_type_id
        WHERE c.object_id = t.object_id
        ORDER BY c.column_id
        FOR XML PATH(''), TYPE
    ).value('.', 'nvarchar(max)')
    + ')' AS Definition
FROM sys.tables t
WHERE t.name IN ('$(($Tables -join "','"))')
ORDER BY SchemaName, ObjectName
"@ -ObjectType "Tables"
}

# --- Types ---
if ($Types -and $Types.Count -gt 0) {
    Export-SqlObjects -Query @"
SELECT 
    SCHEMA_NAME(t.schema_id) AS SchemaName,
    t.name AS ObjectName,
    CASE 
        WHEN t.is_table_type = 1 THEN
            'CREATE TYPE [' + SCHEMA_NAME(t.schema_id) + '].[' + t.name + '] AS TABLE' + CHAR(13) + CHAR(10) + 
            '(' + CHAR(13) + CHAR(10) +
            (
                SELECT 
                    '    [' + c.name + '] ' + 
                    UPPER(tp.name) +
                    CASE 
                        WHEN tp.name IN ('varchar', 'char', 'varbinary', 'binary', 'nvarchar', 'nchar')
                            THEN '(' + 
                                CASE WHEN c.max_length = -1 THEN 'MAX' ELSE CAST(
                                    CASE WHEN tp.name LIKE 'n%' THEN c.max_length / 2 ELSE c.max_length END AS VARCHAR(5)
                                ) END + ')'
                        WHEN tp.name IN ('decimal', 'numeric')
                            THEN '(' + CAST(c.precision AS VARCHAR(5)) + ',' + CAST(c.scale AS VARCHAR(5)) + ')'
                        ELSE ''
                    END +
                    CASE WHEN c.is_nullable = 0 THEN ' NOT NULL' ELSE ' NULL' END + 
                    CASE WHEN c.column_id < (SELECT MAX(column_id) FROM sys.columns c0 WHERE c0.object_id = tt.type_table_object_id) THEN ',' ELSE '' END +
                    CHAR(13) + CHAR(10)
                FROM sys.table_types tt
                JOIN sys.columns c ON c.object_id = tt.type_table_object_id
                JOIN sys.types tp ON c.user_type_id = tp.user_type_id
                WHERE tt.user_type_id = t.user_type_id
                ORDER BY c.column_id
                FOR XML PATH(''), TYPE
            ).value('.', 'nvarchar(max)') + 
            ')'
        ELSE
            'CREATE TYPE [' + SCHEMA_NAME(t.schema_id) + '].[' + t.name + '] FROM ' +
            UPPER(bt.name) +
            CASE 
                WHEN bt.name IN ('varchar', 'char', 'varbinary', 'binary', 'nvarchar', 'nchar')
                    THEN '(' + 
                        CASE WHEN t.max_length = -1 THEN 'MAX' ELSE CAST(
                            CASE WHEN bt.name LIKE 'n%' THEN t.max_length / 2 ELSE t.max_length END AS VARCHAR(5)
                        ) END + ')'
                WHEN bt.name IN ('decimal', 'numeric')
                    THEN '(' + CAST(t.precision AS VARCHAR(5)) + ',' + CAST(t.scale AS VARCHAR(5)) + ')'
                ELSE ''
            END +
            CASE WHEN t.is_nullable = 0 THEN ' NOT NULL' ELSE ' NULL' END
    END AS Definition
FROM sys.types t
LEFT JOIN sys.types bt ON t.system_type_id = bt.system_type_id AND bt.user_type_id = bt.system_type_id
WHERE t.is_user_defined = 1 AND t.name IN ('$(($Types -join "','"))')
ORDER BY SchemaName, ObjectName
"@ -ObjectType "Types"
}
}

$connection.Close()
