<#
.SYNOPSIS
    This script backs up or restores User-Defined Tables (UDT) and User-Defined Fields (UDF) in SAP Business One using the DI Server.
.DESCRIPTION
    The script connects to the SAP Business One DI Server, logs in using provided credentials, and either exports or imports UDT and UDF definitions to/from XML files.
.EXAMPLE
    .\Backup-SAP.ps1 -Action Export -BackupFolder "C:\Backup" -CompanyDB "MyCompanyDB"

    This example exports UDT and UDF definitions to the specified output directory.
    .\Backup-SAP.ps1 -Action Import -BackupFolder "C:\Backup" -CompanyDB "MyCompanyDB"
    This example imports UDT and UDF definitions from the specified output directory.
#>
param (
    [Parameter(Mandatory = $false)]
    [string]$BackupFolder = "data",
    [Parameter(Mandatory = $false)]
    [ValidateSet("Export", "Import")]
    [string]$Action = "Export",
    [Parameter(Mandatory = $true)]
    [string]$CompanyDB,
    [Parameter(Mandatory = $false)]
    [string]$DbServer = "localhost",
    [Parameter(Mandatory = $false)]
    [string]$DbType = "dst_MSSQL2019",
    [Parameter(Mandatory = $false)]
    [string]$LicenseServer = "localhost:30000",
    [Parameter(Mandatory = $false)]
    [string]$Language = "ln_English",
    [Parameter(Mandatory = $false)]
    [System.Management.Automation.PSCredential]$DbCredential = (Get-Credential -Message "Enter database credentials"),
    [Parameter(Mandatory = $false)]
    [System.Management.Automation.PSCredential]$CompanyCredential = (Get-Credential -Message "Enter company credentials"),
    [Parameter(Mandatory = $false)]
    [switch]$DryRun,
    [Parameter(Mandatory = $false)]
    [string[]]$Tables = @(),
    [Parameter(Mandatory = $false)]
    [string[]]$FieldsInTables = @()
)

# Create the output directory if it doesn't exist
if (-not (Test-Path -Path $BackupFolder)) {
    New-Item -ItemType Directory -Path $BackupFolder
}

# Get the script directory and dot-source the commons file
$ScriptDir = Split-Path -Parent $MyInvocation.MyCommand.Definition
$CommonsPath = Join-Path $ScriptDir "DIS-Commons.ps1"

if (Test-Path $CommonsPath) {
    . $CommonsPath
} else {
    Write-Error "DIS-Commons.ps1 not found at: $CommonsPath"
    Write-Error "Please ensure DIS-Commons.ps1 is in the same directory as this script."
    exit 1
}

$node = DIS_CreateObject

# Connect to SAP Business One
$SessionID = DIS_Login -node $node -DbServer $DbServer -CompanyDB $CompanyDB -DbType $DbType -DbCredential $DbCredential -CompanyCredential $CompanyCredential -LicenseServer $LicenseServer -Language $Language -UseTrusted $false -ShowDetails $false

switch ($Action) {
    "Export" {
        # Export user-defined tables using SQL query
        $TableName_Filter = $Tables | ForEach-Object { "('@'+TableName) LIKE '$_'" }
        $OUTB_Filter = @()
        $OUTB_Filter += "(" + ($TableName_Filter -join " OR ") + ")"
        $UserTablesSQL = @"
<dis:ExecuteSQL xmlns:dis="http://www.sap.com/SBO/DIS">
    <DoQuery>SELECT * FROM OUTB WHERE $($OUTB_Filter -join " AND ")</DoQuery>
</dis:ExecuteSQL>
"@
        $UserTables = DIS_Send -node $node -SessionID $SessionID -Body $UserTablesSQL -ShowDetails $true
        $UserTablesXML = [xml]$UserTables
        $UserTablesXML.Save("$BackupFolder\UserTables.xml")

        Write-Host "Response: $($UserTablesXML.OuterXml)"
        Write-Host "User-defined tables exported to $BackupFolder\UserTables.xml"
        
        # Export user-defined fields using SQL query
        $TableName_Filter = ($Tables + $FieldsInTables) | ForEach-Object { "TableID LIKE '$_'" }
        $CUFD_Filter = @()
        $CUFD_Filter += "(" + ($TableName_Filter -join " OR ") + ")"
        $UserFieldsSQL = @"
<dis:ExecuteSQL xmlns:dis="http://www.sap.com/SBO/DIS">
    <DoQuery>SELECT * FROM CUFD WHERE $($CUFD_Filter -join " AND ")</DoQuery>
</dis:ExecuteSQL>
"@
        $UserFields = DIS_Send -node $node -SessionID $SessionID -Body $UserFieldsSQL -ShowDetails $true
        $UserFieldsXML = [xml]$UserFields
        $UserFieldsXML.Save("$BackupFolder\UserFields.xml")

        Write-Host "Response: $($UserFieldsXML.OuterXml)"
        Write-Host "User-defined fields exported to $BackupFolder\UserFields.xml"
    }
    "Import" {

        # Import user-defined tables and fields using AddObject and UpdateObject with Object=oUserTables according to UserTables.xml and UserFields.xml
        # UserTablesMD structure: <?xml version="1.0"?><env:Envelope xmlns:env="http://www.w3.org/2003/05/soap-envelope"><env:Body><GetBusinessObjectTemplateResponse xmlns="http://www.sap.com/SBO/DIS"><BOM><BO><AdmInfo><Object>oUserTables</Object></AdmInfo><QueryParams><TableName/></QueryParams><UserTablesMD><row><TableName/><TableDescription/><TableType/><Archivable/><ArchiveDateField/></row></UserTablesMD></BO></BOM></GetBusinessObjectTemplateResponse></env:Body></env:Envelope>
        # UserFieldsMD structure: <?xml version="1.0"?><env:Envelope xmlns:env="http://www.w3.org/2003/05/soap-envelope"><env:Body><GetBusinessObjectTemplateResponse xmlns="http://www.sap.com/SBO/DIS"><BOM><BO><AdmInfo><Object>oUserFieldsMD</Object></AdmInfo><QueryParams><TableName/><FieldID/></QueryParams><UserFieldsMD><row><Name/><Type/><Size/><Description/><SubType/><LinkedTable/><DefaultValue/><TableName/><EditSize/><Mandatory/><LinkedUDO/><LinkedSystemObject/></row></UserFieldsMD><ValidValuesMD><row><Value/><Description/></row></ValidValuesMD></BO></BOM></GetBusinessObjectTemplateResponse></env:Body></env:Envelope>
        # UserTables.xml has OUTB structure, UserFields.xml has CUFD structure, so they need to be converted to UserTablesMD and UserFieldsMD structures respectively using loops
        <# OUTB row:
            <row>
              <TableName>ACT_CAD_ATT</TableName>
              <Descr>XML adatok - melléklet</Descr>
              <TblNum>14</TblNum>
              <ObjectType>5</ObjectType>
              <UsedInObj />
              <LogTable />
              <Archivable>N</Archivable>
              <ArchivDate />
              <DisplyMenu>Y</DisplyMenu>
              <ApplyPerm>N</ApplyPerm>
            </row>
              CUFD row:
            <row>
              <TableID>@ACT_CAD_ATT</TableID>
              <FieldID>0</FieldID>
              <AliasID>HeadCode</AliasID>
              <Descr>HeadCode</Descr>
              <TypeID>N</TypeID>
              <EditType />
              <SizeID>11</SizeID>
              <EditSize>11</EditSize>
              <Dflt />
              <NotNull>N</NotNull>
              <IndexID>N</IndexID>
              <RTable />
              <RField>0</RField>
              <Action />
              <Sys>N</Sys>
              <DfltDate />
              <RelUDO />
              <ValidRule />
              <RelSO />
              <RThrdPTab />
              <RThrdPFld />
            </row>
        #>
        # Gather existing tablenames from database
        $ExistingTables = @()
        $GetExistingTablesSQL = @"
<dis:ExecuteSQL xmlns:dis="http://www.sap.com/SBO/DIS">
    <DoQuery>
        SELECT TableName FROM OUTB
    </DoQuery>
</dis:ExecuteSQL>
"@
        $ExistingTablesResponse = DIS_Send -node $node -SessionID $SessionID -Body $GetExistingTablesSQL -ShowDetails $false
        $ExistingTablesXML = [xml]$ExistingTablesResponse
        foreach ($row in $ExistingTablesXML.Envelope.Body.ExecuteSQLResponse.BOM.BO.OUTB.row) {
            $ExistingTables += $row.TableName
        }

        # Gather existing field AliasID as associated array by TableID
        $ExistingFields = @{}
        $GetExistingFieldsSQL = @"
<dis:ExecuteSQL xmlns:dis="http://www.sap.com/SBO/DIS">
    <DoQuery>
        SELECT TableID, AliasID FROM CUFD
    </DoQuery>
</dis:ExecuteSQL>
"@
        $ExistingFieldsResponse = DIS_Send -node $node -SessionID $SessionID -Body $GetExistingFieldsSQL -ShowDetails $false
        $ExistingFieldsXML = [xml]$ExistingFieldsResponse
        foreach ($row in $ExistingFieldsXML.Envelope.Body.ExecuteSQLResponse.BOM.BO.CUFD.row) {
            $tableID = $row.TableID
            if (-not $ExistingFields.ContainsKey($tableID)) {
                $ExistingFields[$tableID] = @()
            }
            $ExistingFields[$tableID] += $row.AliasID
        }

        $UserTablesXML = [xml](Get-Content "$BackupFolder\UserTables.xml")
        $UserFieldsXML = [xml](Get-Content "$BackupFolder\UserFields.xml")

        $NewTables = @()
        $NewFields = @()

        $UserTablesMD = "<UserTablesMD></UserTablesMD>"
        foreach ($row in $UserTablesXML.Envelope.Body.ExecuteSQLResponse.BOM.BO.OUTB.row) {
            if ($ExistingTables -contains $row.TableName) {
                # Write-Host "Table $($row.TableName) already exists. Skipping creation."
                continue
            }
            $UserTablesMD += "<row>"
            $UserTablesMD += "<TableName>$($row.TableName)</TableName>"
            $UserTablesMD += "<TableDescription>$($row.Descr)</TableDescription>"
            $UserTablesMD += "<TableType>$($row.TableType)</TableType>"
            $UserTablesMD += "<Archivable>$($row.Archivable)</Archivable>"
            $UserTablesMD += "<ArchiveDateField>$($row.ArchivDate)</ArchiveDateField>"
            $UserTablesMD += "</row>"
            $NewTables += $row.TableName
        }
        $UserFieldsMD = "<UserFieldsMD></UserFieldsMD>"
        foreach ($row in $UserFieldsXML.Envelope.Body.ExecuteSQLResponse.BOM.BO.CUFD.row) {
            if ($ExistingFields.ContainsKey($row.TableID) -and ($ExistingFields[$row.TableID] -contains $row.AliasID)) {
                # Write-Host "Field $($row.AliasID) in table $($row.TableID) already exists. Skipping creation."
                continue
            }
            $UserFieldsMD += "<row>"
            $UserFieldsMD += "<Name>$($row.AliasID)</Name>"
            $UserFieldsMD += "<Type>$($row.TypeID)</Type>"
            $UserFieldsMD += "<Size>$($row.SizeID)</Size>"
            $UserFieldsMD += "<Description>$($row.Descr)</Description>"
            $UserFieldsMD += "<SubType></SubType>"
            $UserFieldsMD += "<LinkedTable>$($row.RTable)</LinkedTable>"
            $UserFieldsMD += "<DefaultValue>$($row.Dflt)</DefaultValue>"
            $UserFieldsMD += "<TableName>$($row.TableID.TrimStart('@'))</TableName>"
            $UserFieldsMD += "<EditSize>$($row.EditSize)</EditSize>"
            $UserFieldsMD += "<Mandatory>$($row.NotNull)</Mandatory>"
            $UserFieldsMD += "<LinkedUDO>$($row.RelUDO)</LinkedUDO>"
            $UserFieldsMD += "<LinkedSystemObject>$($row.RelSO)</LinkedSystemObject>"
            $UserFieldsMD += "</row>"
            $NewFields += "$($row.TableID):$($row.AliasID)"
        }
        $AddUserTablesBody = @"
<dis:AddObject xmlns:dis="http://www.sap.com/SBO/DIS">
    <AdmInfo>
        <Object>oUserTables</Object>
    </AdmInfo>
    $UserTablesMD
</dis:AddObject>
"@

        $AddUserFieldsBody = @"
<dis:AddObject xmlns:dis="http://www.sap.com/SBO/DIS">
    <AdmInfo>
        <Object>oUserFieldsMD</Object>
    </AdmInfo>
    $UserFieldsMD
</dis:AddObject>
"@

        Write-Host "New tables: $($NewTables.Count)/$($UserTablesXML.Envelope.Body.ExecuteSQLResponse.BOM.BO.OUTB.row.Count)"
        Write-Host "New fields: $($NewFields.Count)/$($UserFieldsXML.Envelope.Body.ExecuteSQLResponse.BOM.BO.CUFD.row.Count)"

        if ($DryRun) {
            Write-Host "DRY RUN MODE: No changes will be made to the database."
            if ($NewTables.Count -gt 0) {
                Write-Host "Would create $($NewTables.Count) new user-defined tables."
                Write-Host "Tables:`n  $($NewTables -join "`n  ")"
            }
            if ($NewFields.Count -gt 0) {
                Write-Host "Would create $($NewFields.Count) new user-defined fields."
                Write-Host "Fields:`n  $($NewFields -join "`n  ")"
            }
        } else {
            if ($NewTables.Count -gt 0) {
                $AddUserTablesResponse = DIS_Send -node $node -SessionID $SessionID -Body $AddUserTablesBody -ShowDetails $true
                Write-Host "AddObject UserTables Response: $($AddUserTablesResponse)"
            } else {
                Write-Host "No new user-defined tables to import."
            }
            if ($NewFields.Count -gt 0) {
                $AddUserFieldsResponse = DIS_Send -node $node -SessionID $SessionID -Body $AddUserFieldsBody -ShowDetails $true
                Write-Host "AddObject UserFields Response: $($AddUserFieldsResponse)"
            } else {
                Write-Host "No new user-defined fields to import."
            }
        }
    }
}

# Logout from SAP Business One
DIS_Logout -node $node -SessionID $SessionID -ShowDetails $false > $null
# Write-Host "Logged out from SAP Business One DI Server."
